/**
 * Angio計算ツール統合アプリ - 共通データ管理システム
 * 3つのアプリ間でデータを共有するためのライブラリ
 */

class AngioSharedData {
    constructor() {
        this.storageKey = 'angio-shared-data';
        this.initializeData();
    }

    // 初期データ構造
    getDefaultData() {
        return {
            // 共通設定
            settings: {
                cra_max_angle: 45,
                cau_max_angle: 45,
                lao_max_angle: 180,
                rao_max_angle: 180,
                precision: 1,
                language: 'ja',
                theme: 'medical-blue'
            },

            // 患者・症例情報
            patient: {
                id: '',
                name: '',
                procedure_date: '',
                vessel_type: '',
                lesion_location: '',
                clinical_indication: ''
            },

            // 最後に使用された角度設定
            lastAngles: {
                horizontal_type: 'LAO',
                horizontal_angle: 0,
                vertical_type: 'CRA',
                vertical_angle: 0
            },

            // 病変血管ベクトル設定
            vesselVector: {
                input_mode: 'individual', // 'individual' or 'preset'
                preset_number: 1,
                horizontal_type: 'LAO',
                horizontal_angle: 0,
                vertical_type: 'CRA',
                vertical_angle: 0,
                description: '',
                last_updated: null
            },

            // 設定角度情報（規定データ）
            presetAngles: {
                1: { horizontal_type: 'RAO', horizontal_angle: 50, vertical_type: 'CRA', vertical_angle: 5, description: '右冠動脈' },
                2: { horizontal_type: 'RAO', horizontal_angle: 100, vertical_type: 'CAU', vertical_angle: 75, description: 'LAD近位部' },
                3: { horizontal_type: 'LAO', horizontal_angle: 145, vertical_type: 'CAU', vertical_angle: 10, description: 'LAD中間部' },
                4: { horizontal_type: 'LAO', horizontal_angle: 80, vertical_type: 'CAU', vertical_angle: 15, description: 'LAD遠位部' },
                5: { horizontal_type: 'LAO', horizontal_angle: 95, vertical_type: 'CRA', vertical_angle: 10, description: 'LCX近位部' },
                6: { horizontal_type: 'LAO', horizontal_angle: 55, vertical_type: 'CAU', vertical_angle: 5, description: 'LCX中間部' },
                7: { horizontal_type: 'LAO', horizontal_angle: 30, vertical_type: 'CAU', vertical_angle: 30, description: 'LCX遠位部' },
                9: { horizontal_type: 'LAO', horizontal_angle: 80, vertical_type: 'CAU', vertical_angle: 30, description: 'RCA近位部' },
                10: { horizontal_type: 'LAO', horizontal_angle: 65, vertical_type: 'CAU', vertical_angle: 40, description: 'RCA中間部' },
                11: { horizontal_type: 'LAO', horizontal_angle: 140, vertical_type: 'CAU', vertical_angle: 25, description: 'RCA遠位部' },
                12: { horizontal_type: 'LAO', horizontal_angle: 90, vertical_type: 'CAU', vertical_angle: 15, description: 'PDA' },
                13: { horizontal_type: 'LAO', horizontal_angle: 155, vertical_type: 'CAU', vertical_angle: 55, description: 'PLB' },
                14: { horizontal_type: 'LAO', horizontal_angle: 70, vertical_type: 'CAU', vertical_angle: 35, description: 'OM' }
            },

            // 各アプリの最新結果
            results: {
                'angio-90deg': null,
                'angio-ivus': null,
                'angio-optimize': null
            },

            // 計算履歴
            history: [],

            // アプリ間の設定同期
            sync: {
                auto_sync_angles: true,
                auto_sync_limits: true,
                save_history: true
            }
        };
    }

    // データ初期化
    initializeData() {
        const stored = localStorage.getItem(this.storageKey);
        if (!stored) {
            this.data = this.getDefaultData();
            this.save();
        } else {
            try {
                this.data = JSON.parse(stored);
                // 新しいフィールドがある場合はマージ
                this.data = this.mergeWithDefaults(this.data);
            } catch (e) {
                console.error('Failed to parse stored data:', e);
                this.data = this.getDefaultData();
                this.save();
            }
        }
    }

    // デフォルト値とマージ
    mergeWithDefaults(storedData) {
        const defaults = this.getDefaultData();
        return this.deepMerge(defaults, storedData);
    }

    // オブジェクトの深いマージ
    deepMerge(target, source) {
        const result = { ...target };
        for (const key in source) {
            if (source[key] && typeof source[key] === 'object' && !Array.isArray(source[key])) {
                result[key] = this.deepMerge(target[key] || {}, source[key]);
            } else {
                result[key] = source[key];
            }
        }
        return result;
    }

    // データ保存
    save() {
        try {
            localStorage.setItem(this.storageKey, JSON.stringify(this.data));
            this.notifyChange();
        } catch (e) {
            console.error('Failed to save data:', e);
        }
    }

    // 設定の取得/設定
    getSetting(key) {
        return this.data.settings[key];
    }

    setSetting(key, value) {
        this.data.settings[key] = value;
        this.save();
    }

    // 制限値の管理（CRA/CAU/LAO/RAO）
    getAngleLimits() {
        return {
            cra_max: this.data.settings.cra_max_angle,
            cau_max: this.data.settings.cau_max_angle,
            lao_max: this.data.settings.lao_max_angle,
            rao_max: this.data.settings.rao_max_angle
        };
    }

    setAngleLimits(cra_max, cau_max, lao_max, rao_max) {
        this.data.settings.cra_max_angle = cra_max;
        this.data.settings.cau_max_angle = cau_max;
        if (lao_max !== undefined) {
            this.data.settings.lao_max_angle = lao_max;
        }
        if (rao_max !== undefined) {
            this.data.settings.rao_max_angle = rao_max;
        }
        this.save();
    }

    // 最後の角度設定の管理
    getLastAngles() {
        return { ...this.data.lastAngles };
    }

    setLastAngles(angles) {
        this.data.lastAngles = { ...this.data.lastAngles, ...angles };
        this.save();
    }

    // 病変血管ベクトルの管理
    getVesselVector() {
        return { ...this.data.vesselVector };
    }

    setVesselVector(vector) {
        this.data.vesselVector = {
            ...this.data.vesselVector,
            ...vector,
            last_updated: new Date().toISOString()
        };
        this.save();
    }

    // 病変血管ベクトルから3Dベクトルを計算
    calculateVesselVector3D() {
        const vessel = this.data.vesselVector;
        return AngioSharedData.calculateVector(
            vessel.horizontal_type,
            vessel.horizontal_angle,
            vessel.vertical_type,
            vessel.vertical_angle
        );
    }

    // 規定角度データの管理
    getPresetAngles() {
        return { ...this.data.presetAngles };
    }

    setPresetAngle(number, angleData) {
        this.data.presetAngles[number] = { ...angleData };
        this.save();
    }

    getPresetAngle(number) {
        return this.data.presetAngles[number] ? { ...this.data.presetAngles[number] } : null;
    }

    // 規定データを病変血管ベクトルに適用
    applyPresetToVesselVector(presetNumber) {
        const preset = this.getPresetAngle(presetNumber);
        if (preset) {
            this.setVesselVector({
                input_mode: 'preset',
                preset_number: presetNumber,
                horizontal_type: preset.horizontal_type,
                horizontal_angle: preset.horizontal_angle,
                vertical_type: preset.vertical_type,
                vertical_angle: preset.vertical_angle,
                description: preset.description
            });
        }
    }

    // 患者情報の管理
    getPatientInfo() {
        return { ...this.data.patient };
    }

    setPatientInfo(patientInfo) {
        this.data.patient = { ...this.data.patient, ...patientInfo };
        this.save();
    }

    // 計算結果の保存
    saveResult(appName, result) {
        this.data.results[appName] = {
            timestamp: new Date().toISOString(),
            data: result
        };

        // 履歴に追加
        if (this.data.sync.save_history) {
            this.data.history.unshift({
                app: appName,
                timestamp: new Date().toISOString(),
                result: result
            });

            // 履歴は最大50件まで
            if (this.data.history.length > 50) {
                this.data.history = this.data.history.slice(0, 50);
            }
        }

        this.save();
    }

    // 計算結果の取得
    getResult(appName) {
        return this.data.results[appName];
    }

    // 計算履歴の取得
    getHistory(limit = 10) {
        return this.data.history.slice(0, limit);
    }

    // データのエクスポート
    exportData() {
        return {
            ...this.data,
            export_timestamp: new Date().toISOString(),
            version: '1.0'
        };
    }

    // データのインポート
    importData(importedData) {
        try {
            if (importedData.version) {
                delete importedData.version;
                delete importedData.export_timestamp;
            }

            this.data = this.mergeWithDefaults(importedData);
            this.save();
            return true;
        } catch (e) {
            console.error('Failed to import data:', e);
            return false;
        }
    }

    // データのリセット（制限角度と設定角度情報は保持）
    resetData() {
        const preservedSettings = {
            cra_max_angle: this.data.settings.cra_max_angle,
            cau_max_angle: this.data.settings.cau_max_angle,
            lao_max_angle: this.data.settings.lao_max_angle,
            rao_max_angle: this.data.settings.rao_max_angle
        };
        const preservedPresetAngles = { ...this.data.presetAngles };

        this.data = this.getDefaultData();

        // 保持すべきデータを復元
        this.data.settings.cra_max_angle = preservedSettings.cra_max_angle;
        this.data.settings.cau_max_angle = preservedSettings.cau_max_angle;
        this.data.settings.lao_max_angle = preservedSettings.lao_max_angle;
        this.data.settings.rao_max_angle = preservedSettings.rao_max_angle;
        this.data.presetAngles = preservedPresetAngles;

        // その他の角度データをリセット（CRA 0度に設定）
        this.data.lastAngles = {
            horizontal_type: 'LAO',
            horizontal_angle: 0,
            vertical_type: 'CRA',
            vertical_angle: 0
        };
        this.data.vesselVector = {
            input_mode: 'individual',
            preset_number: 1,
            horizontal_type: 'LAO',
            horizontal_angle: 0,
            vertical_type: 'CRA',
            vertical_angle: 0,
            description: '',
            last_updated: null
        };

        this.save();
    }

    // 変更通知（他のアプリがリスンできるよう）
    notifyChange() {
        window.dispatchEvent(new CustomEvent('angio-data-changed', {
            detail: { data: this.data }
        }));
    }

    // イベントリスナーの登録
    onDataChange(callback) {
        window.addEventListener('angio-data-changed', (event) => {
            callback(event.detail.data);
        });
    }

    // 共通数学関数
    static degToRad(deg) {
        return deg * Math.PI / 180;
    }

    static radToDeg(rad) {
        return rad * 180 / Math.PI;
    }

    // 角度の正規化
    static normalizeAngle(angle) {
        while (angle > 180) angle -= 360;
        while (angle < -180) angle += 360;
        return angle;
    }

    // ベクトル計算（共通関数）
    static calculateVector(hType, hAngle, vType, vAngle) {
        const hRad = this.degToRad(hType === 'RAO' ? -hAngle : hAngle);
        const vRad = this.degToRad(vType === 'CAU' ? -vAngle : vAngle);

        return {
            x: Math.cos(vRad) * Math.sin(hRad),
            y: -Math.sin(vRad),
            z: Math.cos(vRad) * Math.cos(hRad)
        };
    }

    // ベクトルの内積
    static dotProduct(v1, v2) {
        return v1.x * v2.x + v1.y * v2.y + v1.z * v2.z;
    }

    // ベクトルの正規化
    static normalizeVector(v) {
        const magnitude = Math.sqrt(v.x * v.x + v.y * v.y + v.z * v.z);
        return {
            x: v.x / magnitude,
            y: v.y / magnitude,
            z: v.z / magnitude
        };
    }
}

// グローバルインスタンス
window.AngioSharedData = AngioSharedData;
window.angioData = new AngioSharedData();

// 使用例とヘルパー関数
window.AngioHelpers = {
    // 角度入力フィールドの作成
    createAngleInputGroup: function(id, label, defaultType = 'LAO', defaultAngle = 0) {
        return `
            <div class="angle-input-group" data-angle-id="${id}">
                <label>${label}</label>
                <div class="angle-controls">
                    <select id="${id}-type">
                        <option value="LAO" ${defaultType === 'LAO' ? 'selected' : ''}>LAO</option>
                        <option value="RAO" ${defaultType === 'RAO' ? 'selected' : ''}>RAO</option>
                        <option value="CRA" ${defaultType === 'CRA' ? 'selected' : ''}>CRA</option>
                        <option value="CAU" ${defaultType === 'CAU' ? 'selected' : ''}>CAU</option>
                    </select>
                    <input type="number" id="${id}-angle" value="${defaultAngle}" min="0" max="90" step="1">
                    <span>°</span>
                </div>
            </div>
        `;
    },

    // フォームデータを共通データに同期
    syncFormToSharedData: function(formElement, appName) {
        const formData = new FormData(formElement);
        const data = {};
        for (let [key, value] of formData.entries()) {
            data[key] = value;
        }
        window.angioData.saveResult(appName, data);
    },

    // 共通データをフォームに読み込み
    loadSharedDataToForm: function(formElement, appName) {
        const result = window.angioData.getResult(appName);
        if (result && result.data) {
            for (let [key, value] of Object.entries(result.data)) {
                const element = formElement.querySelector(`[name="${key}"]`);
                if (element) {
                    element.value = value;
                }
            }
        }
    }
};

// アプリ間のデータ同期を支援する関数
window.syncAngioData = {
    // 各アプリから呼び出せる共通データアクセス関数
    getAngleLimits: () => window.angioData.getAngleLimits(),
    setAngleLimits: (cra, cau) => window.angioData.setAngleLimits(cra, cau),
    getLastAngles: () => window.angioData.getLastAngles(),
    setLastAngles: (angles) => window.angioData.setLastAngles(angles),
    getVesselVector: () => window.angioData.getVesselVector(),
    setVesselVector: (vector) => window.angioData.setVesselVector(vector),
    calculateVesselVector3D: () => window.angioData.calculateVesselVector3D(),
    getPresetAngles: () => window.angioData.getPresetAngles(),
    getPresetAngle: (number) => window.angioData.getPresetAngle(number),
    setPresetAngle: (number, angleData) => window.angioData.setPresetAngle(number, angleData),
    applyPresetToVesselVector: (presetNumber) => window.angioData.applyPresetToVesselVector(presetNumber),
    getPatientInfo: () => window.angioData.getPatientInfo(),
    saveResult: (appName, result) => window.angioData.saveResult(appName, result),

    // 共通計算関数
    degToRad: AngioSharedData.degToRad,
    radToDeg: AngioSharedData.radToDeg,
    calculateVector: AngioSharedData.calculateVector,
    normalizeAngle: AngioSharedData.normalizeAngle
};